package gsd.common;

import gsd.impl.Buffers;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.UUID;


public class NodeID extends ID{

	private UUID id;
	private InetSocketAddress listenPort;
	
	public NodeID(InetSocketAddress socketAddr) {
		this.id = UUID.randomUUID();
		this.listenPort = socketAddr;
	}
	
	public NodeID(InetSocketAddress socketAddr, UUID id) {
		this.id = id;
		this.listenPort = socketAddr;
	}
	
	public NodeID(NodeID nodeID) {
		this.id = nodeID.getUUID();
		this.listenPort = nodeID.getSocketAddress();
	}

	public UUID getUUID() {
		return this.id;
	}
	
	public InetSocketAddress getSocketAddress() {
		return this.listenPort;
	}
	
	static public ByteBuffer writeNodeIdToBuffer(NodeID id) {
		ByteBuffer msg = ByteBuffer.allocate(22);
		InetSocketAddress isa = id.getSocketAddress();
		UUID uuid = id.getUUID();
		
		msg.put(isa.getAddress().getAddress());
		msg.putShort((short) isa.getPort());
		
		msg.putLong(uuid.getMostSignificantBits());
		msg.putLong(uuid.getLeastSignificantBits());
		
		msg.flip();
		return msg;
	}
	
	static public void writeNodeIdToBuffer(ByteBuffer buf, NodeID node) {
		InetSocketAddress isa = node.getSocketAddress();
		UUID uuid = node.getUUID();
		
		buf.put(isa.getAddress().getAddress());
		buf.putShort((short) isa.getPort());
		
		buf.putLong(uuid.getMostSignificantBits());
		buf.putLong(uuid.getLeastSignificantBits());
	}
	
	public static NodeID readNodeIDFromBuffer(ByteBuffer[] msg) {
		return readNodeIDFromBuffer(Buffers.sliceCompact(msg, 22));
	}
	
	public static NodeID readNodeIDFromBuffer(ByteBuffer buf) {
		int port = 0;
		byte[] dst = new byte[4];
		InetAddress ia = null;
		
		buf.get(dst, 0, dst.length);
		port = ((int) buf.getShort()) & 0xffff;
		try {
			ia = InetAddress.getByAddress(dst);
		} catch (UnknownHostException e) {
			// We are sure that this does not happen, as the
			// byte array is created with the correct size.
		}
		
		return new NodeID(new InetSocketAddress(ia, port), new UUID(buf.getLong(), buf.getLong()));        
	}
	
	public boolean equals(Object o) {
		if(o != null && o instanceof NodeID) {
			NodeID nid = (NodeID) o;
			return this.listenPort.equals(nid.listenPort) && this.id.equals(nid.id);
		} else {
			return false;
		}		
	}
	
	public String toString() {
		return this.listenPort.toString() + "::" + this.id.toString();
	}

	public void setUUID(UUID id) {
		this.id = id;
	}
	
	public int hashCode() {
		return this.id.hashCode();
	}
}
